﻿namespace Microsoft.Samples.PlanMyNight.Web
{
    using System;
    using System.ComponentModel.Composition;
    using System.Security.Principal;
    using System.Web;
    using System.Web.Profile;
    using System.Web.Security;
    using Microsoft.Samples.PlanMyNight.Data;
    using Microsoft.Samples.PlanMyNight.Entities;

    public interface IFormsAuthentication
    {
        void SignIn(string userName, bool createPersistentCookie);
        void SignOut();
    }

    public interface IMembershipService
    {
        bool ValidateUser(string userName, string password);

        MembershipCreateStatus CreateUser(string userName, string password, string email);

        bool ChangePassword(string userName, string oldPassword, string newPassword);

        UserProfile GetCurrentProfile();

        UserProfile CreateProfile(string userName);

        void UpdateProfile(UserProfile profile);
    }

    public static class UserProfileExtensions
    {
        public static UserProfile GetAsUserProfile(this ProfileBase profileBase)
        {
            if (profileBase == null)
            {
                return null;
            }

            var user = new UserProfile();
            user.FullName = profileBase.GetPropertyValue("FullName") as string;
            user.State = profileBase.GetPropertyValue("State") as string;
            user.City = profileBase.GetPropertyValue("City") as string;
            user.PreferredActivityTypeId = (int)profileBase.GetPropertyValue("PreferredActivityTypeId");
            return user;
        }

        public static void UpdateFromUserProfile(this ProfileBase profileBase, UserProfile user)
        {
            profileBase.SetPropertyValue("FullName", user.FullName);
            profileBase.SetPropertyValue("State", user.State);
            profileBase.SetPropertyValue("City", user.City);
            profileBase.SetPropertyValue("PreferredActivityTypeId", user.PreferredActivityTypeId ?? 0);
        }
    }

    [Export(typeof(IFormsAuthentication))]
    [PartCreationPolicy(CreationPolicy.NonShared)]
    public class FormsAuthenticationService : IFormsAuthentication
    {
        private readonly HttpContextBase httpContext;

        public FormsAuthenticationService() :
            this(new HttpContextWrapper(HttpContext.Current))
        {
        }

        public FormsAuthenticationService(HttpContextBase httpContext)
        {
            this.httpContext = httpContext;
        }

        public void SignIn(string userName, bool createPersistentCookie)
        {
            var ticket = new FormsAuthenticationTicket(userName, false, (int)FormsAuthentication.Timeout.TotalMinutes);
            var hash = FormsAuthentication.Encrypt(ticket);
            this.httpContext.User = new GenericPrincipal(new FormsIdentity(ticket), new[] { "user" });
            this.httpContext.Response.Cookies.Add(new HttpCookie(FormsAuthentication.FormsCookieName, hash));
        }

        public void SignOut()
        {
            if (this.httpContext.Response.Cookies[FormsAuthentication.FormsCookieName] != null)
            {
                this.httpContext.Response.Cookies[FormsAuthentication.FormsCookieName].Expires = DateTime.Now.AddDays(-1);
            }
        }
    }

    [Export(typeof(IMembershipService))]
    [PartCreationPolicy(CreationPolicy.NonShared)]
    public class AccountMembershipService : IMembershipService
    {
        private const string ProfileCacheContainer = "default";

        private readonly TimeSpan ProfileCacheTimeOut = TimeSpan.FromMinutes(20);

        private readonly MembershipProvider provider;

        private readonly ICachingProvider cacheProvider;

        private readonly HttpContextBase httpContext;

        [ImportingConstructor]
        public AccountMembershipService(ICachingProvider cachingProvider)
            : this(Membership.Provider, cachingProvider, new HttpContextWrapper(HttpContext.Current))
        {
        }

        public AccountMembershipService(MembershipProvider provider, ICachingProvider cacheProvider, HttpContextBase httpContext)
        {
            this.provider = provider;
            this.cacheProvider = cacheProvider;
            this.httpContext = httpContext;
        }

        public int MinPasswordLength
        {
            get
            {
                return this.provider.MinRequiredPasswordLength;
            }
        }

        public bool ValidateUser(string userName, string password)
        {
            return this.provider.ValidateUser(userName, password);
        }

        public MembershipCreateStatus CreateUser(string userName, string password, string email)
        {
            MembershipCreateStatus status;
            this.provider.CreateUser(userName, password, email, null, null, true, null, out status);
            return status;
        }

        public bool ChangePassword(string userName, string oldPassword, string newPassword)
        {
            MembershipUser currentUser = this.provider.GetUser(userName, true /* userIsOnline */);
            return currentUser.ChangePassword(oldPassword, newPassword);
        }

        public UserProfile GetCurrentProfile()
        {
            var cacheKey = this.httpContext.User.Identity.Name;
            var profile = this.cacheProvider.Get(ProfileCacheContainer, cacheKey) as UserProfile;
            if (profile != null)
            {
                return profile;
            }

            profile = this.httpContext.Profile.GetAsUserProfile();
            this.cacheProvider.Add(ProfileCacheContainer, cacheKey, profile, this.ProfileCacheTimeOut);
            return profile;
        }

        public UserProfile CreateProfile(string userName)
        {
            return ProfileBase.Create(userName, true).GetAsUserProfile();
        }

        public void UpdateProfile(UserProfile user)
        {
            var profile = this.httpContext.Profile;
            profile.UpdateFromUserProfile(user);
            profile.Save();

            var cacheKey = profile.UserName;
            this.cacheProvider.Remove(ProfileCacheContainer, cacheKey);
        }
    }
}
